/*************************************************************************************************************
Main.c

This is the main source code file for the GPS Clock Sweep for sweep hand clocks.

Geoff Graham
	Version 1.0   July 2009
	
IMPORTANT:
For up to date errata, notes and new firmware go to http://geoffg.net/GPS_Synchronised_Clock.html

Development Environment
	 - Microchip MPLAB IDE Version 8.0 or higher (www.microchip.com)
	 - CCS C compiler V4.0 or higher (www.ccsinfo.com)
	          - OR -
	   Hi-Tech C Compiler PICC Pro V9.6 Lite or full version (www.htsoft.com)
	   
	   The HiTech compiler will complain about constructs that are only used by the CCS compiler.
	   To prevent this navigate to Project->Build Options->Project->Compiler and set Warnings to 1

Program summary:
	Most of the work is done in the timer 1 interrupt.  This interrupt occurs on the leading and trailing edge
	of each clock pulse.  In between interrupts the cpu is put to sleep.
	
	The variable ClkErr is used to track the difference between where the clock's hands are and where they should be.
	Each count of ClkErr represents 1/8th of a second fast or slow.  The variable ClkPos is used track the actual
	position of the clock's hands.  It counts upwards in units of 1/8th of a second and resets to zero ot the 30 minute
	mark (ie, the hour or half hour).
	
	When ClkErr is positive the hands are ahead of the currect position (fast) and the code in the timer 1 interrupt
	will select longer pulses to slow the clock down.  If negative, short pulses will be selected.  After 16 long
	or short pulses timer 1 will decrement or increment ClkErr.  Thus ClkErr will trend towards zero.
	
	Following a GPS sync the time is converted into 8ths of a second and converted to be zero at the hour or half hour.
	It is then compared to ClkPos and any difference is written into ClkErr.  The difference is also divided into the
	total time (in 8ths of a second) recorded by the crystal since the last sync (XtalCnt).  This result is counted 
	down to zero every 8ths of a second and, when zero is reached, ClkErr is incremented or decremented accordingly.
	This continuously corrects any errors in the crystal frequency.
	
	Main() initialises everything, gets the time from the GPS, waits for the next hour/half hour to arrive and then
	drops into a neverending loop where waits to be woken up by an interrupt.  It then loops through code that looks 
	for error events (low battery, etc) and GPS synchronisation events.

***************************************************************************************************************/
#ifdef HI_TECH_C
    #define _XTAL_FREQ 			4000000
    #include "HiTech-16F88.h"
    // configuration fuses
    // internal oscillator with I/O on the clock pins, MCLR on pin 4 enabled, and everything else disabled
    __CONFIG(UNPROTECT & DEBUGDIS & LVPDIS & BORDIS & MCLREN & PWRTDIS & WDTDIS & INTIO);
    __CONFIG(FCMDIS & IESODIS);
#endif

#ifdef __PCM__
    #include <16F88.h>
    #include "CCS-16F88.h"
    // configuration fuses
    // internal oscillator with I/O on the clock pins, MCLR on pin 4 enabled, and everything else disabled
    #fuses INTRC_IO,NOIESO,NOPUT,NOFCMEN,NOWDT,NOCPD,NOWRT,NOPROTECT,MCLR,NOBROWNOUT,NOLVP,NODEBUG
    	// Clock Related Fuses (select one):
    	//        LP  Low-Power Crystal
    	//        XT  Crystal/Resonator
    	//        HS  High-Speed Crystal/Resonator
    	//        RC  External Resistor-Capacitor (RC) with FOSC/4 output on RA6
    	//        RC_IO  External Resistor-Capacitor with I/O on RA6
    	//        INTRC - Internal Oscillator with FOSC/4 output on RA6 and I/O on RA7
    	//        INTRC_IO - Internal Oscillator with I/O on RA6 and RA7
    	//        EC_IO  External Clock with I/O on RA6.
    	// Int/Ext Clock Switchover:     IESO,NOIESO
    	// Fail-Safe Clock Monitor:      FCMEN,NOFCMEN
    	// Watch Dog Timer:              WDT, NOWDT
    	// Protect Data Memory:          CPD,NOCPD
    	// Protect Code:                 PROTECT,NOPROTECT
    	// Prog mem write protected      WRT,NOWRT
    	// Use MCLR Pin for Reset:       MCLR, NOMCLR
    	// Power-up Timer:               PUT,NOPUT
    	// Brown-out Detect:             BROWNOUT,NOBROWNOUT
    	// Low-Volt Programming Enable:  LVP, NOLVP
    	// In-Circuit Debugger Mode:     DEBUG,NODEBUG
    	// CCP1 Pin Selection:           CCPB0, CCPB3

    #use delay(clock=4000000)
#endif

#define CHECK_CHECKSUM                  // Validate the checksum on data from the GPS.  Comment out to save 40 bytes

#define TRISA_VAL 			0b00100111	// value of TRISA, this MUST agree with the #defines below
#define TRISB_VAL 			0b11111100	// value of TRISB, this MUST agree with the #defines below
#define ANSEL_VAL           0b00000100  // value of ANSEL, this selects analog inputs and must match below

#define	CLK				    RA0			// output/input - step pulse to the clock movement
#define	CLK_OLD 			RA1			// input  - was used to drive clock stepper motor, now not used
#define REF_VOLT			RA2			// input  - analog input from the MAX756 reference voltage
#define LED					RA3			// output - simple diagnostic indicator
#define GPS_PWR				RA4			// output - low will apply power to the GPS
#define MCLR				RA5			// input  - reset circuit used to delay CPU start by 1 second
#define GPS_TX				RA6			// output - transmit data to the GPS (not used)
#define PC_TX				RA7			// output - RS232 transmit data for setup

#define UNASSIGNED1			RB0			// output
#define UNASSIGNED2			RB1			// output
#define GPS_RX_UART			RB2			// input  - receive data from the GPS to hardware UART
#define PC_RX				RB3			// input  - RS232 receive data for setup
#define GPS_RX				RB4			// input  - receive data from the GPS for firmware, now not used
#define BUTTON				RB5			// input  - pushbutton input
#define T1OSC1				RB6			// input  - reserved for timer1 oscillator crystal
#define T1OSC2				RB7			// input  - reserved for timer1 oscillator crystal

#define CLK_FLOAT             TRISA0    // direction control
#define CLK_ANALOG            ANS0      // turn the clock output pin into an analog input


#define Mark1				{ UNASSIGNED1 = 1; UNASSIGNED1 = 0; }
#define Mark1double			{ UNASSIGNED1 = 1; UNASSIGNED1 = 0; UNASSIGNED1 = 1; UNASSIGNED1 = 0; }
#define Mark2				{ UNASSIGNED2 = 1; UNASSIGNED2 = 0; }
#define Mark2double			{ UNASSIGNED2 = 1; UNASSIGNED2 = 0; UNASSIGNED2 = 1; UNASSIGNED2 = 0; }
#define Mark3				{ UNASSIGNED3 = 1; UNASSIGNED3 = 0; }
#define Mark3double			{ UNASSIGNED2 = 1; UNASSIGNED2 = 0; UNASSIGNED2 = 1; UNASSIGNED2 = 0; }



/************************************************************************************************
Configuration defines
************************************************************************************************/
#define VERSION             "1.0"

#define SECONDS_IN_HOUR		3600

#define	GET_GPS_TIMEOUT		240			// number of seconds to wait for a GPS lock
#define GPS_RETRY_TIME		4			// number of hours before retrying a failed GPS lock
#define MAX_GPS_ERRORS		10			// number of times the GPS can fail to lock before we give up

#define LOW_BATT 			166			// low battery threshold (166 = 2V) Note: Higher nbr means lower volts
#define MAX_BATT_LOWCOUNT	4			// number of battery low indications before halting
#define MAX_GPS_ERRORCOUNT	10			// number of gps failures to find the satellites before halting
#define MAX_GPS_DEADCOUNT	10			// number of gps failures to communicate before halting

#define PULSE_WIDTH_STD     4           // pulse width of a standard clock pulse (in units of 8mS)
#define PULSE_WAIT_STD      4           // wait time between pulses for a standard clock pulse (in units of 8mS)
#define PULSE_WIDTH_FAST    3           // pulse width of a standard clock pulse (in units of 8mS)
#define PULSE_WAIT_FAST     5           // wait time between pulses for a standard clock pulse (in units of 8mS)

#ifdef HI_TECH_C
    #define TIME4800BAUD    181         // delay per bit
#endif

#ifdef __PCM__
    #define TIME4800BAUD    193         // delay per bit
#endif



/************************************************************************************************
Global memory locations
************************************************************************************************/

// all variables marked volatile may be changed in the clock interrupt
// and special care must be taken in accessing them as they could change while being used
volatile sint16 StartCnt;               // counter used at the initial startup while waiting for the time to reach the hour
volatile sint16 ClkPos = 0;             // logical position (zero on the hour and half hour) in units of eights of a second
volatile sint16 ClkErr = 0;             // error in seconds/8 - +ive if hands are fast, -ive if hands are slow
volatile sint32 XtalCnt = 0;            // how many seconds/8 since the last gps sync
sint32 XtalError = 0;                   // counter used to correct for errors in the crystal's timing

volatile bit T1Done = false;            // indicates when timer1 has finished one 8th of a second
volatile uint8 NewSecCnt = 0;           // count to determine when a second has passed

sint16 GpsTime;						    // the time returned by the GPS in seconds in the hour
volatile sint16 XtalTime = 0;           // the gps time when we last synced
uint8 GpsCount = 1;                     // timer (in half hours) to the next sync
uint8 BatteryLowCount = 0;				// number of times the battery voltage has been low
uint8 GpsDeadCount = 0;                 // number of times that the gps has not responded
uint8 GpsErrorCount = 0;				// countdown for failure to get a GPS lock
volatile uint8 GpsTimeout;              // used to determine when we have timed out getting data from the gps

                                        // used to save the various timing parameters for timer 1
uint16  T1PulseFast, T1PulseMed, T1PulseSlow, T1WaitFast, T1WaitMed, T1WaitSlow;



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// initialise the eeprom configuration data,  these are the default configuration values:
// to save space, we also store strings in the eeprom.  each string (listed below) is terminated with a zero byte

#define MSG_START	        3	                // location in the eeprom where the strings start
#define GPS_SYNC_INTERVAL   0                   // location of the value in the eeprom
#define PULSE_WIDTH         1                   // ditto
#define RUN_TIME            2

#define M_title             0
#define M_update            1
#define M_now               2
#define M_2set              3
#define M_pulse             4
#define M_cmd               5
#define M_stop              6
#define M_colon             7
#define M_enter             8
#define M_error             9
#define M_hours             10


// this is for the HiTech C Compiler
#ifdef HI_TECH_C

// this the data stored in the eeprom.  unfortunately the hi-tech c compiler
// does not provide an easier way to store strings
// the first two entries are GPS sync intervals in hours and standard pulse width
__EEPROM_DATA(44, 34, 0, 0x000D, 0x000D, 0x0047, 0x0050, 0x0053); 
__EEPROM_DATA(0x0020, 0x0043, 0x006C, 0x006F, 0x0063, 0x006B, 0x0020, 0x0028); 
__EEPROM_DATA(0x0073, 0x0077, 0x0065, 0x0065, 0x0070, 0x0020, 0x0068, 0x0061); 
__EEPROM_DATA(0x006E, 0x0064, 0x0073, 0x0029, 0x0020, 0x0031, 0x002E, 0x0030); 
__EEPROM_DATA(0x000D, 0x000D, 0x0020, 0x0031, 0x0020, 0x003D, 0x0020, 0x0053); 
__EEPROM_DATA(0x0065, 0x0074, 0x0020, 0x0000, 0x0047, 0x0050, 0x0053, 0x0020); 
__EEPROM_DATA(0x0075, 0x0070, 0x0064, 0x0061, 0x0074, 0x0065, 0x0020, 0x0069); 
__EEPROM_DATA(0x006E, 0x0074, 0x0065, 0x0072, 0x0076, 0x0061, 0x006C, 0x0020); 
__EEPROM_DATA(0x0028, 0x0068, 0x0072, 0x0073, 0x0029, 0x0000, 0x0020, 0x0020); 
__EEPROM_DATA(0x0020, 0x0020, 0x0028, 0x006E, 0x006F, 0x0077, 0x0020, 0x0000); 
__EEPROM_DATA(0x000D, 0x0020, 0x0032, 0x0020, 0x003D, 0x0020, 0x0053, 0x0065); 
__EEPROM_DATA(0x0074, 0x0020, 0x0000, 0x0063, 0x006C, 0x006F, 0x0063, 0x006B); 
__EEPROM_DATA(0x0020, 0x0070, 0x0075, 0x006C, 0x0073, 0x0065, 0x0020, 0x0077); 
__EEPROM_DATA(0x0069, 0x0064, 0x0074, 0x0068, 0x0020, 0x0028, 0x006D, 0x0053); 
__EEPROM_DATA(0x0065, 0x0063, 0x0029, 0x0020, 0x0000, 0x000D, 0x000D, 0x0020); 
__EEPROM_DATA(0x0052, 0x0020, 0x003D, 0x0020, 0x0052, 0x0075, 0x006E, 0x0020); 
__EEPROM_DATA(0x0063, 0x006C, 0x006F, 0x0063, 0x006B, 0x0020, 0x0066, 0x006F); 
__EEPROM_DATA(0x0072, 0x0020, 0x0061, 0x0020, 0x0066, 0x0069, 0x0078, 0x0065); 
__EEPROM_DATA(0x0064, 0x0020, 0x0074, 0x0069, 0x006D, 0x0065, 0x000D, 0x0020); 
__EEPROM_DATA(0x0051, 0x0020, 0x003D, 0x0020, 0x0051, 0x0075, 0x0069, 0x0074); 
__EEPROM_DATA(0x000D, 0x000D, 0x0043, 0x006F, 0x006D, 0x006D, 0x0061, 0x006E); 
__EEPROM_DATA(0x0064, 0x003A, 0x0020, 0x0000, 0x000D, 0x0050, 0x0072, 0x0065); 
__EEPROM_DATA(0x0073, 0x0073, 0x0020, 0x0053, 0x0065, 0x0074, 0x0075, 0x0070); 
__EEPROM_DATA(0x0020, 0x0062, 0x0075, 0x0074, 0x0074, 0x006F, 0x006E, 0x0020); 
__EEPROM_DATA(0x0074, 0x006F, 0x0020, 0x0073, 0x0074, 0x006F, 0x0070, 0x0020); 
__EEPROM_DATA(0x0065, 0x0061, 0x0072, 0x006C, 0x0079, 0x0020, 0x0000, 0x003A); 
__EEPROM_DATA(0x0020, 0x0000, 0x000D, 0x0045, 0x006E, 0x0074, 0x0065, 0x0072); 
__EEPROM_DATA(0x0020, 0x0000, 0x000D, 0x0049, 0x006E, 0x0076, 0x0061, 0x006C); 
__EEPROM_DATA(0x0069, 0x0064, 0x0020, 0x006E, 0x0075, 0x006D, 0x0062, 0x0065); 
__EEPROM_DATA(0x0072, 0x000D, 0x0020, 0x0000, 0x006E, 0x0075, 0x006D, 0x0062); 
__EEPROM_DATA(0x0065, 0x0072, 0x0020, 0x006F, 0x0066, 0x0020, 0x006D, 0x0069); 
__EEPROM_DATA(0x006E, 0x0075, 0x0074, 0x0065, 0x0073, 0x0000, 0x0000, 0x0000);

#endif


// this is for the CCS C Compiler
#ifdef __PCM__

#rom  0x2100 = {44,						 // GPS sync intervals in hours
				34,                      // standard pulse width
				0                       // default run time in hours
			}                 

// to save space we also store strings in the eeprom
#rom  0x2103 =  {"\r\rGPS Clock (sweep hands) "VERSION"\r\r 1 = Set ",                  // 0
                 "GPS update interval (hrs)",                                           // 1
                 "    (now ",                                                           // 2
				 "\r 2 = Set ",                                                         // 3
				 "clock pulse width (mSec) ",                                           // 4
				 "\r\r R = Run clock for a fixed time\r Q = Quit\r\rCommand: ",         // 5
				 "\rPress Setup button to abort ",                                      // 6
				 ": ",                                                                  // 7
				 "\rEnter ",                                                            // 8
				 "\rInvalid number\r ",                                                 // 9
				 "number of minutes"                                                    // 10
				}				

#endif



/************************************************************************************************
Declare functions
************************************************************************************************/
void PC_putc(uint8 c);
void GetTimeFromGPS();	
bit GetGPSData(bit FullData);
void GpsSkipComma(uint8);
uint8 GetGpsDataByte();
void CheckGpsChar(uint8 c);
void CheckGpsHexChar(uint8 c);
uint8 GetGpsChar(void);
uint8 GPS_getch(void);
bit GPS_kbhit(void);
void GPS_SetupUART(void);
void FlashLED(uint8 nbr);
void HaltClock();                                    
uint8 BattVoltage();
void delay_SECx10(uint8 d);
void DoConfiguration();
int FindString(uint8 msg_nbr);
void CopyString(uint8 msg_nbr);
void OutputRomString(uint8 msg_nbr);
void PrintMenuItem(uint8 txt1, uint8 txt2, uint8 txt3);
void GetNumber(sint8 *value, sint8 lo, sint8 hi, sint8 point, uint8 txt1, uint8 txt2, uint8 txt3, uint8 txt4);
uint8 PC_getc(void);
void StepSecondHand(uint8 pulse, uint8 width);
void OutputString(uint8 *s);
void OutputInteger(uint8 n);
void GetInteger(uint8 max, uint8 min, uint8 save);
uint16 CalcT1Parameters(sint8 offset, bit pulse);



/************************************************************************************************
Timer1 interrupt
This does two functions:
 - Countdowns while waiting for the time to reach the hour when the clock is first started
 - Creates the pulse to step the clock mechanism.  An interrupt occurs on both the leading and
   trailing edges of the pulse (the PIC is in sleep in between).
************************************************************************************************/
#ifdef HI_TECH_C
    void interrupt pic_isr(void) {
#endif

#ifdef __PCM__
    #int_TIMER1
    void  TIMER1_isr(void) {
#endif
	
    static sint8 ClkErrSubCnt = 0;
    static uint8 T1State = 0;		                                // variable defining our position in the timer1 state machine
    static uint16 T1Count;                                          // used to keep track of timer1 seconds
    uint8 T1L_now;                                                  // used to detect when TMR1L has changed
    uint8 T1L_value;                                                // used to save the value to be written to TMR1L
    
    #ifdef HI_TECH_C
    	if(!((TMR1IE)&&(TMR1IF)))  return;			                // abort if not the timer1 interrupt
    #endif
     if(StartCnt) {
        // this countdown is only used at the initial startup while waiting for the time to reach the hour or half hour
        StartCnt--;
        TMR1H = 128;
        T1Done = true;                                              // signal that a second has passed
    } else {
        // the main pulse generating section
        // pulses are generated in groups of two (one positive, one negative)
        // eight of these groups will move the clock's hands by one second
        if((T1State & 0x01) == 0) {                                 // if state = 0 or 2, ie the leading edge of a pulse
            CLK = (T1State == 0);                                   // start of s pulse, set high or low depending on the state
            CLK_FLOAT = OFF;                                        // enable the pin as an output
            if(ClkErr > 0) {                                        // clock is running fast
                TMR1H = WordHiByte(T1PulseSlow);
                T1L_value = WordLoByte(T1PulseSlow);
                T1Count -= T1PulseSlow;
            } else if(ClkErr < 0) {                                 // clock is running slow
                TMR1H = WordHiByte(T1PulseFast);
                T1L_value = WordLoByte(T1PulseFast); 
                T1Count -= T1PulseFast;
            } else {  // ClkErr == 0                                // clock is spot on
                TMR1H = WordHiByte(T1PulseMed);
                T1L_value = WordLoByte(T1PulseMed);
                T1Count -= T1PulseMed;
            }
            T1State++;
        } else {                                                    // if state = 1 or 3, ie the trailing edge of a pulse
            CLK_FLOAT = ON;                                         // end of the pulse, so set the pin to high impedance
            if(ClkErr > 0) {                                        // clock fast
                TMR1H = WordHiByte(T1WaitSlow);
                T1L_value = WordLoByte(T1WaitSlow);
                T1Count -= T1WaitSlow;
                if(T1State == 3)
                    if(--ClkErrSubCnt == -16) {                     // this counts 16 sub corrections for one click of ErrCnt
                        ClkErrSubCnt = 0;                   
                        ClkErr--;                                   // one click of ClkErr equals 1/8th of a second correction
                    }    
            } else if(ClkErr < 0) {                                 // clock slow
                TMR1H = WordHiByte(T1WaitFast);
                T1L_value = WordLoByte(T1WaitFast); 
                T1Count -= T1WaitFast;
                if(T1State == 3)
                    if(++ClkErrSubCnt == 16) {                      // this counts 16 sub corrections for one click of ErrCnt
                        ClkErrSubCnt = 0;
                        ClkErr++;                                   // one click of ClkErr equals 1/8th of a second correction
                    }    
            } else {  // ClkErr == 0                                // clock accurate
                TMR1H = WordHiByte(T1WaitMed);
                T1L_value = WordLoByte(T1WaitMed);
                T1Count -= T1WaitMed;
            }                        
            T1State++;
        }
        
        // synchronise writing to timer1's low register - this waits until timer1 changes, that means we then have 
        // approx 30uS (30 instructions) to write the new value without clashing with the next timer1 increment.
        // We add the value to the current count in TMR1L, this has the effect of taking into account whatever value 
        // that has already accumulated in TMR1L.  Note that we do not check for overflow so it is assumed the 
        // granularity of the values in T1L_value will always greater than the accumulated value in TMR1L.  This is 
        // true as T1L_value uses increments of 1 mSec which (for easy arithmetic) we assume has the value of 0x20.
        T1L_now = TMR1L;
        while(TMR1L == T1L_now);                                    // loop on the spot until a new count occurs
        TMR1L += T1L_value;                                         // set the new value - easy!
        
        // accumulate the time (in 8ths of a second) according to timer1 since the last sync
        // this is used to determine the error of timer1
        while(T1Count & 0xf000) {                                   // while counter is greater than 1/8th of a second
            T1Count -= 16 * 256;                                    // decrement by 1/8th of a second
            XtalCnt++;                                              // count the crystal's version of a second (in 1/8ths)
            if(++XtalTime == (SECONDS_IN_HOUR/2) * 8) XtalTime = 0; // this tracks what the hands would show without correction
        } 
        
        // this processing happens at the end of a group of four pulses (ie, every 1/8th of a second)
        if(T1State == 4) {
            if(++ClkPos == (SECONDS_IN_HOUR/2) * 8) ClkPos = 0;     // increment the clock's logical position and wrap if we have reached the hour or half hour
            
            if(++NewSecCnt == 8) {                                  // determine if we have reached a new second
                NewSecCnt = 0;
                if(GpsTimeout) GpsTimeout--;                        // used for timing out gps communications
            }
            T1State = 0;
            T1Done = true;
        }
    }
          
#ifdef HI_TECH_C
	TMR1IF = false;													// clear the interrupt flag
#endif
}




/**********************************************************************************************
Main program
**********************************************************************************************/

void main() {
	sint32 ErrCountdown = 0;
	uint8 i;
	
	OSCCON = 0b01101110;								            // run this chip at 4MHz using the internal clock
   
	GPS_PWR = 1;										            // set the GPS power control before we enable the outputs
	LED = 0;                                                        // and the led
	ANSEL = ANSEL_VAL;                                              // config analogue inputs
	
	// set pin directions
	TRISA = TRISA_VAL;
	TRISB = TRISB_VAL;
    
    #ifdef __PCM__
        setup_timer_1(T1_EXTERNAL);
        T1CON = 0b00001111;								            // timer1 clock is a crystal oscillator running at 32.768KHz
    #endif
    
    #ifdef HI_TECH_C
        T1CON = 0b00001111;								            // timer1 clock is a crystal oscillator running at 32.768KHz
    #endif
    
	FlashLED(1);										            // Flash the LED once
    if(!BUTTON) DoConfiguration();					                // do configuration if the user has held down the button

    // calculate the parameters for timer1.
    // we do it once here and save the values to make the code within the timer interrupt simpler
    T1PulseFast = CalcT1Parameters(-2, true);
    T1PulseMed = CalcT1Parameters(0, true);
    T1PulseSlow = CalcT1Parameters(+2, true);
    T1WaitFast = CalcT1Parameters(-2, false);
    T1WaitMed = CalcT1Parameters(0, false);
    T1WaitSlow = CalcT1Parameters(+2, false);

	GPS_PWR = 0;										            // power up the GPS
    delay_SECx10(15);
	if(BattVoltage() < 20)  HaltClock();				            // check that the MAX756 has powered up
	FlashLED(2);										            // Flash the LED twice to show progress

    GPS_SetupUART();                                                // init the UART to 4800 baud async
    delay_SECx10(15);
    if(!GetGPSData(false)) HaltClock();				                // check that the GPS is sending something out
	FlashLED(3);										            // Flash the LED thrice to show that the GPS is alive

	delay_SECx10(15);
    if(!GetGPSData(true)) HaltClock();				                // get a valid time from the GPS
	GPS_PWR = 1;									    	        // turn off the GPS
	
	StartCnt = SECONDS_IN_HOUR/2 - GpsTime;                         // figure out how many seconds before we start running
    //StartCnt = 3;		
    TMR1H = 255; TMR1L = 250;							            // set timer1 to interrupt almost immediately
    #ifdef __PCM__
        PEIE = ON;                                                  // enable interrupts from peripherals
    	enable_interrupts(INT_TIMER1);					            // and start processing timer interrupt
        enable_interrupts(GLOBAL);						            // GO !!
    #endif
	
    #ifdef HI_TECH_C
        PEIE = ON;                                                  // enable interrupts from peripherals
    	TMR1IE = ON;									            // enable timer1 interrupts
    	GIE = ON;										            // enable GLOBAL Interrupts
    #endif

	FlashLED(4);										            // Flash the LED to show that the GPS got valid data
   
    i = 5;                                                          // wait 5 seconds before we start the slow flash
    while(StartCnt) {                                               // waiting for the time to reach the hour or half hour
        if(!BUTTON)                                                 // step around the dial if the user has pressed the button
            StepSecondHand(eeprom_read(PULSE_WIDTH), 63 - eeprom_read(PULSE_WIDTH));
        else if(T1Done) {                                           // if a second has passed
            T1Done = false;
            if(--i == 0) {                                          // count 3 seconds
                i = 3;
                LED = 1; mSec(100); LED = 0;                        // and flash the LED
            }
        }
    }            

    T1Done = false;                                                 // prepare for the main loop

    // main processing loop
    // we do a trip around the loop eight times every second
	while(forever) {                                                // now start the main loop
		while(!T1Done) sleep();                                     // sleep until the next second/8
        T1Done = false;
       
		if(GpsDeadCount >= MAX_GPS_DEADCOUNT && ClkPos == 50*30*8) 
			HaltClock();								            // GPS is not responding - halt at 10 minutes before the hour or half hour

		if(GpsErrorCount >= MAX_GPS_ERRORCOUNT && ClkPos == 55*30*8) 
			HaltClock();								            // GPS cannot get the time - halt at 5 minutes before the hour or half hour

		if(BatteryLowCount >= MAX_BATT_LOWCOUNT && ClkPos == 0)
			HaltClock();								            // battery low, stop at exactly on the hour or half hour

        if(ErrCountdown == 0) ErrCountdown = XtalError;             // note all counters are in units of one 8ths of a second
        if(XtalError > 0)
            if(--ErrCountdown == 0) ClkErr++;                       // compensate for a fast running crystal
        if(XtalError < 0)
            if(++ErrCountdown == 0) ClkErr--;                       // and for a slow one

        if(ClkPos == 0) GpsCount--;
	    if(GpsCount == 0 || !BUTTON)				                // synch on the hour/half or when the user pressed button
		    GetTimeFromGPS();
    }
}
	



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Perform all the work involved in getting the time from the GPS and correcting for any errors 
//
void GetTimeFromGPS() {			
    static bit GotFirstGPSLock = false;                             // true if we have got our first time sync
	sint16 t;	
	
	GPS_PWR = 0;        								            // turn on the GPS
	mSec(250);										                // wait for the EM-408 module to draw full current
	mSec(250);										                // the HiTech compiler forces us to call a function
    mSec(100);										                // with 255 as the maximum argument
    
    if(BattVoltage() > LOW_BATT)                                    // check battery. Note: Higher number means lower voltage
		BatteryLowCount++;
	else
		BatteryLowCount = 0;
		
	GpsCount = GPS_RETRY_TIME * 2;		                            // this is the default timeout if we have an error
		
	if(!GetGPSData(false)) {							            // check that the GPS is alive
		GpsDeadCount++;
		GPS_PWR = 1;							                    // turn off the GPS
		return;											            // and skip the sync
	}
	GpsDeadCount = 0;									            // GPS is working
	

	if(!GetGPSData(true)) {						                    // get the time from the GPS
	    GPS_PWR = 1;								                // turn off the GPS
		GpsErrorCount++;
		return;											            // and abort
	}
	GpsErrorCount = 0;                                              // no errors
	GpsTime = GpsTime << 3;                                         // convert to eigths of a second for the rest of our calculations
	
	GPS_PWR = 1;								                    // turn off the GPS
	FlashLED(1);										            // show that we got good data
	
	// Wait for a new second/8.  This synchronisation is necessary because the interrupt may change variables while 
	// the remainder of this function is also using them.  Following this wait we will have at least 24mS before 
	// the next interrupt occurs and that is plenty of time to complete all the processing in this routine.
	if(TMR0IE) {                                                    // only if the timer is running
	    T1Done = false;
	    while(!T1Done);                                             // wait until we have just completed a new second/8				
	}
	
	// this calculates the error between the uncorrected crystal oscillator time and the gps time
	// note that the -7 in the calculation is the time (7/8 second) between the start of the gps packet and reaching here
	if(GotFirstGPSLock)	{							                // don't bother if this is the first time sync
    	GpsCount = eeprom_read(GPS_SYNC_INTERVAL) << 1;             // setup the next GPS sync
    	t = XtalTime - 7 - GpsTime;	                                // calculate the difference between the clock and gps in seconds/8
    	if(t < (SECONDS_IN_HOUR/4) * 8) t += (SECONDS_IN_HOUR/2) * 8;	// compensate if clock and/or gps has wrapped around the 30 min point
    	if(t > (SECONDS_IN_HOUR/4) * 8) t -= (SECONDS_IN_HOUR/2) * 8;
    } else {
        t = 0;
    	GpsCount = 4;                                               // sync again in two hours to quickly compensate for any crystal error
    	GotFirstGPSLock = true;
    }  
    
    if(t == 0)
		XtalError = 0;                                              // avoid divide by zero error
	else
	    XtalError = XtalCnt/(sint32)t;				                // calculate the error factor between the xtal and the gps
    
	// this calculates the error between the actual position of the clock's hands and the gps
	ClkErr = ClkPos - 7 - GpsTime;
	if(ClkErr < (SECONDS_IN_HOUR/4) * 8) ClkErr += (SECONDS_IN_HOUR/2) * 8;	// compensate if clock and/or gps has wrapped around the 30 min point
	if(ClkErr > (SECONDS_IN_HOUR/4) * 8) ClkErr -= (SECONDS_IN_HOUR/2) * 8;

	XtalCnt = 0;                                                    // same for this one
	XtalTime = GpsTime + 7;
}



/************************************************************************************************
Get data from the GPS module

Arguments:
FullData	If true the function will return only after receiving a valid $GPRMC string
			If false will return after receiving just "$GP" indicating a working GPS

Returns:	True if data received OK.  In that case GpsTime will hold the GPS time in 1/8ths 
            of a second since the start of the nearest hour or half hour.
			False if timeout
					
Format of the NEMA message.  We only use the minutes (mm) and seconds (ss):
    $GPRMC,043356.000,A,3158.7599,S,11552.8689,E,0.24,54.42,101008,,*20
    ====== ======     =                                     ======   ==
    fixed   time   valid                                    date    checksum
    header hhmmss   data                                    ddmmyy
************************************************************************************************/
bit GpsError;
uint8 GpsChecksum;

bit GetGPSData(bit FullData) {
    uint8 m, s;
    #ifdef CHECK_CHECKSUM
    	uint8 ChecksumSave;
    #endif
	
	GpsTimeout = GET_GPS_TIMEOUT;					                // setup the timeout
	
	do {
		GpsError = false;
		while(GetGpsChar() != '$') if(GpsTimeout == 0) return false;

		GpsChecksum = 0;
		CheckGpsChar('G');
		CheckGpsChar('P');
		if(FullData) {                                              // we can skip the rest if just testing the gps is alive
    		CheckGpsChar('R');
    		CheckGpsChar('M');
    		CheckGpsChar('C');
    		CheckGpsChar(',');
    		(void)GetGpsDataByte();                                 // throw away the hours
    		m = GetGpsDataByte();                                   // get the minutes
    		s = GetGpsDataByte();                                   // and seconds
    		GpsSkipComma(1);
    		CheckGpsChar('A');								        // indicates that the GPS has a lock on enough satellites
    		GpsSkipComma(7);
    
            #ifdef CHECK_CHECKSUM
        		do 
        		    ChecksumSave = GpsChecksum;
        		while(GetGpsChar() != '*');                         // skip to the checksum
        		CheckGpsHexChar(ChecksumSave >> 4);                 // first byte of the checksum
        		CheckGpsHexChar(ChecksumSave & 0x0f);               // and the second
            #else
          		CheckGpsChar(',');                                  // just test for a comma - very unlikely that it would fail
            #endif
        }      
	} while(GpsError);									            // only exit if everything OK
	
	if((m >= 30)) m -= 30;                                          // time is tracked to the half hour
        
	GpsTime = (sint16)m * 60 + (sint16)s;                           // calculate the time in seconds into the half hour
	return true;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a number of commas.  Used for skipping over unwanted data
//
void GpsSkipComma(uint8 nbr) {
	while(nbr)
		if(GetGpsChar() == ',') nbr--;
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a data byte consisting of two ascii chars
//
uint8 GetGpsDataByte() {
	uint8 b;
	
	b = (GetGpsChar() - '0') * 10;
	if(GpsError) return 0;
	b += (GetGpsChar() - '0');
	return b;	
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a character and test it against the argument.  Set GpsError if mismatch.
//
void CheckGpsChar(uint8 c) {
	if(GetGpsChar() != c) GpsError = true;
}
	
	

///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get from the GPS input stream a hex character and test it against the argument.  Set GpsError if mismatch.
//
#ifdef CHECK_CHECKSUM
    void CheckGpsHexChar(uint8 c) {
        uint8 ch;
    	ch = GetGpsChar() - '0';						            // get the first byte of the checksum
    	if(ch > 9) ch -= 'A' - 58;						            // convert from hex
    	if(ch != c) GpsError = true;	                            // and test
    }
#endif	


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Get a char from the GPS input stream.  
//	- Checks for timeout and returns zero if exceeded.
//	- Skips the read and returns zero if GpsError already set
//	- Converts the char to uppercase and adds it to the checksum
//
uint8 GetGpsChar(void) {
	uint8 ch;
	
	while(!GPS_kbhit())
		if(GpsError || GpsTimeout == 0) {
			GpsError = true;
			return 0;
		}	
			
	ch = GPS_getch();

	GpsChecksum = GpsChecksum ^ ch;
	if(ch >= 'a') ch -= 'a' - 'A';						            // convert to uppercase
	return ch;
}	
	


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Return a character from the GPS serial interface,  Will timeout after some delay
//
uint8 GPS_getch(void) {
    uint16 timeout;

    for(timeout = 0; timeout < 1000; timeout++) {                   // wait for the char, with timeout
        if(OERR) { CREN = OFF; CREN = ON; }
        if(FERR) {uint8 d; d = RCREG; d = RCREG; d = RCREG; }
        if(RCIF) return RCREG & 0x7f;
    }
    return 0;                                                       // return on timeout
}    



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Tests if a character is received by the GPS serial interface
//
bit GPS_kbhit(void) {
    if(OERR) { CREN = OFF; CREN = ON; }
    if(FERR) {uint8 d; d = RCREG; d = RCREG; d = RCREG; }
    return RCIF;
}

    
///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Sets up the GPS serial interface for 4800 baud async
//
void GPS_SetupUART(void) {
    SPBRG = 51;                                                     // baud rate selector, 51 = 4800 baud
    BRGH = 1;
    CREN = OFF;
    ADDEN = OFF;                                                    // no addressing
    SYNC = OFF;                                                     // async
    RCIE = OFF;                                                     // no interrupts
    TXEN = OFF;                                                     // we don't want to transmit
    SPEN = ON;                                                      // enable serial port
    CREN = ON;                                                      // enable receive
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Flash the LED a number of times
// used for simple diagnostics
//
void FlashLED(uint8 nbr) {
	uint8 i;
	
	for(i = 0 ; i < nbr ; i++) {
		delay_SECx10(2);
		LED = 1;
		delay_SECx10(1);
		LED = 0;
	}
}		


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// This will disable all sections of the MCU and put it to sleep
// It is a low power shutdown (less than 1 microamp), there is no return
//
void HaltClock() {
	TMR1IE = false;								                    // disable timer1 interrupts
	GIE = false;								                    // disable GLOBAL Interrupts
    T1CON = 0b00000000;							                    // disable timer1 and the xtal oscillator
    ADCON0 = 0b00000000;						                    // disable A-D converter
    CLK_FLOAT = ON;                                                 // set the clock output to high impedance
    GPS_PWR = 1;								                    // turn off the GPS
	LED = 0;							                            // turn the LED off
	while(forever) sleep();				                            // and go to sleep forever (until the battery is disconnected)
}




///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Read the reference voltage from the MAX756
// Because the ADC reference is the battery voltage we can use this reading to infer the battery voltage.
// This means that decreasing battery voltage will give increasing readings.  With MAX756 Vref = 1.27V a reading
// of 166 gives a battery is 2.0V  A reading of zero means that the MAX756 is not running.
//
uint8 BattVoltage() {
	uint8 t;

	ANSEL = 0b00000100;								                // Sets AN2 pin to be analogue
	ADCON1 = 0b00000000;							                // ADC 8 bit result, ref is Vdd & Vss, no clock divide
	ADCON0 = 0b11010001;							                // ADC internal clock, channel 2, do not start, ADC is on	
	uSec(200);								                        // wait for the input to settle
	GODONE = true;									                // start the conversion
	while(GODONE);									                // wait for the ADC to reset the bit
	t = ADRESH;										                // read the voltage
	ADON = false;									                // turn off the ADC
	return t;
}	



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// delay for a a number of seconds
// argument is in seconds * 10.  e.g.  an argument of 15 will delay for 1.5 seconds
//
void delay_SECx10(uint8 d) {
	while(d--) mSec(100);
}	




///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// calculate the timing parameters for timer 1
// argument is the offset (is this speedup, slowdown, etc) and pulse which is true for pulse false for wait
//
uint16 CalcT1Parameters(sint8 offset, bit pulse) {
    sint8 t;
    uint16 n;
    
    t = eeprom_read(PULSE_WIDTH) + 1;
    t += offset;
    if(!pulse) t = (64 + (offset * 2)) - t;
    n = t;
    n = n << 5;
    n = (uint16)0xffff - n; n++;                                    // avoid a compiler bug
    return n;
}    



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// step the second hand around the dial.
//
void StepSecondHand(uint8 pulse, uint8 width) {
            CLK = 1; CLK_FLOAT = OFF; mSec(pulse);
            CLK_FLOAT = ON; mSec(width);
            CLK = 0; CLK_FLOAT = OFF; mSec(pulse);
            CLK_FLOAT = ON; mSec(width);
}



/************************************************************************************************
Functions associated with changing the configuration data in EEPROM
Because we are so short of ROM, many strings are stored in EEPROM and functions like
FindString(int msg_nbr) are used to retrieve them.
************************************************************************************************/
//*
///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Print the menu and get input from the user.  Updates the EEPROM accordingly
// Can return to the caller if the user enters Q
//
void DoConfiguration() {
	uint16 i;
	uint8 c;

    while(forever) {    
        OutputRomString(M_title); OutputRomString(M_update); OutputRomString(M_now); OutputInteger(eeprom_read(GPS_SYNC_INTERVAL)); PC_putc(')');
        OutputRomString(M_2set); OutputRomString(M_pulse); OutputRomString(M_now); OutputInteger(eeprom_read(PULSE_WIDTH)); PC_putc(')');
        OutputRomString(M_cmd);
       
    	c = PC_getc(); PC_putc(c); PC_putc('\r'); PC_putc('\n');
    	switch(c) {
        	case '1':	OutputRomString(M_enter); OutputRomString(M_update); OutputRomString(M_colon);    // prompt for the gps update interval
        	            GetInteger(127, 1, GPS_SYNC_INTERVAL);
    				    break;
    				    
        	case '2':	OutputRomString(M_enter); OutputRomString(M_pulse); OutputRomString(M_colon);    // prompt for the level
        	            GetInteger(57, 4, PULSE_WIDTH);
        				break;
        				
        	case 'r':
        	case 'R':	OutputRomString(M_enter); OutputRomString(M_hours); OutputRomString(M_colon); GetInteger(120, 1, RUN_TIME);
        	            if(eeprom_read(RUN_TIME)) OutputRomString(M_stop);
        	            for(i = 0; (i < (uint16)eeprom_read(RUN_TIME) * 60 * 8) && BUTTON; i++)
            	            switch((i >> 6) % 4) {
                	            case 0:
                	            case 2: StepSecondHand(eeprom_read(PULSE_WIDTH), 63 - eeprom_read(PULSE_WIDTH));        // normal speed
                	                    break;
                	            case 1: StepSecondHand(eeprom_read(PULSE_WIDTH) - 2, 58 - eeprom_read(PULSE_WIDTH));    // fast
                	                    break;
                	            case 3: StepSecondHand(eeprom_read(PULSE_WIDTH) + 2, 68 - eeprom_read(PULSE_WIDTH));    // slow
                	                    break;
                            }
                        eeprom_write(RUN_TIME, 0);                                                      // reset for next time
        	            break;
        	            
        	case 'q':
        	case 'Q':	return;
        }
    }
}    
    


///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Output a string stored in the eeprom
//
void OutputRomString(uint8 msg_nbr) {
	uint8 i, c;
	
	for(i = MSG_START ; msg_nbr != 0 ; i++) if(eeprom_read(i) == 0)
		msg_nbr--;													// get the message

	for(; eeprom_read(i) != 0; i++) {				                // output the string
		c = eeprom_read(i);
		PC_putc(c);
		if(c == '\r') PC_putc('\n');
	}
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// output an integer - could be better designed, but it works
//
void OutputInteger(uint8 n) {
    uint8 d;
    d = n/100;
    if(d) PC_putc(d + '0');
    d = (n % 100)/10;
    if(d) PC_putc(d + '0');
    PC_putc((n % 10) + '0');       
}    



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// get an integer from the user and store it in the eeprom
//
void GetInteger(uint8 max, uint8 min, uint8 save) {
    uint8 i = 0;
    uint8 cnt = 0;
    uint8 c;
    
    c = PC_getc();
    while(c != '\r') {                                              // get the char, exit on return
        if(c == 8 && cnt > 0) {                                     // process a backspace
            i = i / 10;
            cnt--;
            PC_putc('\010'); PC_putc(' '); PC_putc('\010');
        }
        if(c >= '0' && c <= '9') {                                  // if numeric
            PC_putc(c);                                             // echo
            i = (i * 10) + (c - '0');                               // save - don't worry about overflow
            cnt++;
        }
        c = PC_getc();
    }
    if(cnt > 0) {
        if(i > max || i < min) 
            OutputRomString(M_error);
        else
            eeprom_write(save, i);                                  // store away
    }
}



///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Return a character from the PC serial interface,  Will wait forever for the char
// Assumes standard RS232 type serial interface (idle is low) and 4800 baud
//
uint8 PC_getc(void) {
	uint8 i, c=0;
	 do {
		while(!PC_RX);							                    // wait for the start bit
		uSec(TIME4800BAUD/2);
	} while(!PC_RX);							                    // check that the start bit is still there
	for(i = 0; i < 8; i++) {
		uSec(TIME4800BAUD);				                            // wait for the next char
		c >>= 1;								                    // shift down the previous bits (LSB is received first)
		if(!PC_RX) c |= 0b10000000;				                    // and put this bit in
	}
	uSec(TIME4800BAUD);			                                    // wait for the last bit to complete (and then some!)
	return c;
}		
	
	

///////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Send a character to the PC serial interface.
// also flashes the LED with each bit sent
// Assumes standard RS232 type serial interface (idle is low) and 4800 baud
//
void PC_putc(uint8 c) {
	uint8 i;
	LED = PC_TX = 1;							                    // send the start bit
	uSec(TIME4800BAUD);					                            // and wait for it to complete
	for(i = 0; i < 8; i++) {
		LED = PC_TX = !(c & 1);					                    // send the next bit
		uSec(TIME4800BAUD);
		c >>= 1;
	}
	LED = PC_TX = 0;							                    // send the stop bit
	uSec(TIME4800BAUD*2);
}


